"use client"
//app/search/page.tsx
import { useState, useEffect, Suspense, useRef, useCallback, useMemo, } from "react"
import { useSearchParams, useRouter } from "next/navigation"
import { Filter, Calendar, Shield, ImageIcon, VideoIcon, GlobeIcon, ChevronDownIcon, Newspaper, ShoppingCart } from "lucide-react"
import { Button } from "@/components/ui/button"
import { Card, CardContent } from "@/components/ui/card"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { LogoWithText } from "@/components/logo"
import { SearchWithAutocomplete } from "@/components/search-with-autocomplete"
import { ImageModal } from "@/components/image-modal"
import { VideoModal } from "@/components/video-modal"
import { SettingsModal } from "@/components/settings-modal"
import { AdCard } from "@/components/ad-card"
import { useSettings } from "@/components/settings-provider"
import { Globe, Video, ChevronDown, Search, Settings } from 'lucide-react';
import AiSearchResult from './AiSearchResult'
import GoogleWebSearchResults from '@/components/ui/GoogleWebSearchResults'
import GoogleImageSearchResults from '@/components/ui/GoogleImageSearchResults'
import Footer from '@/components/ui/Footer'
import { usePreservedResults } from '@/hooks/usePreservedResults'
import { useHistory } from '@/hooks/useHistory'
import WebResults from '@/components/IndzsSearch/WebResults'
import ImageResults from '@/components/IndzsSearch/ImageResults'
import ActionBar from '@/components/ui/ActionBar'

const testData = {
  query: "Future of quantum computing in cybersecurity",
  answer: "Quantum computing is projected to fundamentally alter the cybersecurity landscape by rendering current RSA and ECC encryption methods obsolete through Shor's algorithm. However, the industry is pivoting toward Post-Quantum Cryptography (PQC), utilizing lattice-based and code-based encryption.",
  sources: [
    { title: "Quantum Vulnerabilities", url: "#" },
    { title: "Post-Quantum Standards 2025", url: "#" },
    { title: "Encryption Paradigms", url: "#" },
    { title: "The Decryption Horizon", url: "#" }
  ]
};

interface SearchResult {
  id: string
  title: string
  description: string
  url: string
  favicon?: string
  date?: string
  keywords?: string[]
  safesearch: boolean
  maindomain: boolean
  isAd?: boolean
  // Ad-specific properties
  targetUrl?: string
  company?: string
  adType?: string
}

interface ImageResult {
  id: string
  title: string
  url: string
  imgurl: string
  keywords?: string[]
  favicon?: string
  date?: string
  safesearch: boolean
}

interface VideoResult {
  id: string
  title: string
  thumbnail: string
  url: string
  date?: string
  safesearch: boolean
  channelTitle?: string
  description?: string
}
// const tabs= [
//   { label: "web" },
//   { label: "image" },
//   { label: "video" },
// ];
type Tab = {
  label: string,
  id: string,
  icon: any

};

const tabs: Tab[] = [
  { id: 'web', label: 'Web', icon: Globe },
  { id: 'images', label: 'Images', icon: ImageIcon },
  { id: 'videos', label: 'Videos', icon: Video },
  { id: 'news', label: 'News', icon: Newspaper },
  { id: 'shopping', label: 'Shopping', icon: ShoppingCart },
];
const intentMap: Record<string, { keywords: string[]; weight: number }> = {
  images: {
    keywords: ["image", "images", "photo", "photos", "pic", "pics", "jpg", "png"],
    weight: 3,
  },
  videos: {
    keywords: ["video", "videos", "clip", "clips", "movie", "mp4"],
    weight: 3,
  },
  web: {
    keywords: ["web", "site", "article", "news", "blog", "wiki"],
    weight: 1,
  },

  news: {
    keywords: ["news", "headline", "headlines"],
    weight: 3,
  },

  shopping: {
    keywords: ["shopping", "shop", "buy", "price"],
    weight: 3,
  },
};

const CANONICAL_MAP: Record<string, string> = {
  web: "web",
  images: "images",
  // image: "image",
  videos: "videos",
  // video: "videos",
  news: "news",
  shopping: "shopping",
};

function sortTabsByPrecedence(tabs, scores) {
  return [...tabs].sort((a, b) => {
    const aKey = CANONICAL_MAP[a.id] ?? a.id;
    const bKey = CANONICAL_MAP[b.id] ?? b.id;

    const aScore = scores[aKey] ?? 0;
    const bScore = scores[bKey] ?? 0;

    // Higher score first
    if (bScore !== aScore) {
      return bScore - aScore;
    }

    // Tie → preserve original order
    return tabs.indexOf(a) - tabs.indexOf(b);
  });
}


function fuzzyMatch(word: string, target: string) {
  // console.log(word)
  // console.log(target.includes(word) )
  return target.includes(word) || word.includes(target);
}
function rankTabsByQuery(query: string, tabs: Tab[]) {
  const words = query.toLowerCase().split(/\s+/);
  // console.log(words)
  // console.log(tabs)
  let scores: Record<string, number> = {};
  scores = {
    web: 0,
    images: 0,
    videos: 0,
    news: 0,
    shopping: 0,
  };


  tabs.forEach(tab => (scores[tab.id] = 0));
  for (const word of words) {
    for (const [id, { keywords, weight }] of Object.entries(intentMap)) {
      for (const keyword of keywords) {
        // console.log(keyword)
        if (fuzzyMatch(word, keyword)) {
          scores[id] += weight;
        }
      }
    }
  }
  console.log(scores)

  console.log(sortTabsByPrecedence(tabs, scores));
  return sortTabsByPrecedence(tabs, scores)
}


function SearchPageContent() {
  const searchParams = useSearchParams()
  const router = useRouter()
  const { resultsPerPage, safeSearch: defaultSafeSearch, language, region } = useSettings()
  const isMountedRef = useRef(true)
  const [query, setQuery] = useState(searchParams.get("q") || "")
  const [searchType, setSearchType] = useState(searchParams.get("type") || "web")
  const [safeSearch, setSafeSearch] = useState(defaultSafeSearch ? "on" : "off")
  const [dateFilter, setDateFilter] = useState("all")
  const [results, setResults] = useState<SearchResult[]>([])
  const [imageResults, setImageResults] = useState<ImageResult[]>([])
  const [videoResults, setVideoResults] = useState<VideoResult[]>([])
  const [loading, setLoading] = useState(false)
  const [showFilters, setShowFilters] = useState(false)
  const [error, setError] = useState<string | null>(null)
  const [selectedImage, setSelectedImage] = useState<ImageResult | null>(null)
  const [isModalOpen, setIsModalOpen] = useState(false)
  const [selectedVideo, setSelectedVideo] = useState<VideoResult | null>(null)
  const [isVideoModalOpen, setIsVideoModalOpen] = useState(false)
  const [currentPage, setCurrentPage] = useState(1)
  const [hasMoreResults, setHasMoreResults] = useState(false)
  const [totalResults, setTotalResults] = useState(0)
  const [aiSearch, setAiSearch] = useState(true)
  const [performAiSearch, setPerformAiSearch] = useState(true)
  const { preservedData } = usePreservedResults(`image:${query}`)
  const { saveSearchHistory } = useHistory();
  const [tab, setTabs] = useState([
    { id: 'web', label: 'Web', icon: Globe },
    { id: 'images', label: 'Images', icon: ImageIcon },
    { id: 'videos', label: 'Videos', icon: Video },
    { id: 'news', label: 'News', icon: Newspaper },
    { id: 'shopping', label: 'Shopping', icon: ShoppingCart },
  ])


  const [isInitialWebResultsLoad, setIsInitialWebResultsLoad] = useState(true);

  const [isInitialImageResultsLoad, setIsInitialImageResultsLoad] = useState(true);
  const [isInitialVideoResultsLoad, setIsInitialVideoResultsLoad] = useState(true);
  const [isInitialShoppingResultsLoad, setIsInitialShoppingResultsLoad] = useState(true);
  const [isInitialNewsResultsLoad, setIsInitialNewsResultsLoad] = useState(true);

  useEffect(() => {
    isMountedRef.current = true

    const q = searchParams.get("q")
    const type = searchParams.get("type") || "web"
    if (q) {
      setQuery(q)
      setSearchType(type)

      performSearch(q, type)
    }

    return () => {
      isMountedRef.current = false
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [searchParams.get("q")])

  useEffect(() => {
    if (isMountedRef.current) {
      setSafeSearch(defaultSafeSearch ? "on" : "off")
    }
  }, [defaultSafeSearch])


  const data = useMemo(() => ({ a: 1 }), []);
  const handleClick = useCallback(() => {
    console.log("clicked");
  }, []);


  const performSearch = async (searchQuery: string, type: string, page = 1) => {
    if (!isMountedRef.current) return

    setLoading(true)
    setError(null)

    // setSearchType(type)
    router.push(`/search?q=${encodeURIComponent(query)}&type=${type}`)
    try {
      console.log("[v0] Performing search:", {
        searchQuery,
        type,
        page,
        currentResults: results.length,
        currentImageResults: imageResults.length,
        currentVideoResults: videoResults.length,
      })

      const response = await fetch(
        `/api/search?q=${encodeURIComponent(searchQuery)}&type=${type}&safesearch=${safeSearch}&date=${dateFilter}&page=${page}&limit=${resultsPerPage}&lang=${language}&region=${region}`,
      )

      if (!response.ok) {
        throw new Error(`Search failed: ${response.status}`)
      }

      const data = await response.json()
      console.log(data)
      if (!isMountedRef.current) return

      console.log("[v0] Search response:", {
        page,
        newResultsCount: data.results?.length,
        hasMore: data.hasMore,
        total: data.total,
        adsCount: data.adsCount,
      })

      const newResults = data.results || []
      setTotalResults(data.total || newResults.length)
      setHasMoreResults(data.hasMore || false)

      if (type === "web") {
        if (page === 1) {
          setResults(newResults)
        } else {
          setResults((prevResults) => {
            console.log("[v0] Appending web results:", { prevCount: prevResults.length, newCount: newResults.length })
            return [...prevResults, ...newResults]
          })
        }
        setImageResults([])
        setVideoResults([])
      } else if (type === "images") {
        if (page === 1) {
          setImageResults(newResults)
        } else {
          setImageResults((prevResults) => {
            console.log("[v0] Appending image results:", { prevCount: prevResults.length, newCount: newResults.length })
            return [...prevResults, ...newResults]
          })
        }
        setResults([])
        setVideoResults([])
      } else if (type === "videos") {
        if (page === 1) {
          setVideoResults(newResults)
        } else {
          setVideoResults((prevResults) => {
            console.log("[v0] Appending video results:", { prevCount: prevResults.length, newCount: newResults.length })
            return [...prevResults, ...newResults]
          })
        }
        setResults([])
        setImageResults([])
      }



      setCurrentPage(page)
      const rankedTabs = rankTabsByQuery(searchQuery, tabs);
      console.log(rankedTabs);
      type = rankedTabs[0].id
      console.log(type)
      setTabs(rankedTabs)
      // setSearchType(rankedTabs[0].id)
      // handleTypeChange(type)


    } catch (error) {
      console.error("Search error:", error)
      if (isMountedRef.current) {
        setError(error instanceof Error ? error.message : "Search failed")
        if (page === 1) {
          setResults([])
          setImageResults([])
          setVideoResults([])
        }
      }
    } finally {
      if (isMountedRef.current) {
        setPerformAiSearch(!performAiSearch)
        setLoading(false)
      }
    }
  }

  const handleSearch = (searchQuery: string) => {
    if (searchQuery.trim()) {
      setCurrentPage(1)
      saveSearchHistory({
        query: searchQuery,
        type: "web",
        // filters,
        // resultsState,
      })
      router.push(`/search?q=${encodeURIComponent(searchQuery.trim())}&type=${searchType}`)
    }
  }

  const handleTypeChange = (type: string) => {
    console.log(query)
    setSearchType(type)
    setCurrentPage(1)
    performSearch(query, type)
    // router.push(`/search?q=${encodeURIComponent(query)}&type=${type}`)
  }

  const handleSafeSearchChange = (value: string) => {
    setSafeSearch(value)
    setCurrentPage(1)
    if (query) {
      performSearch(query, searchType, 1)
    }
  }

  const handleDateFilterChange = (value: string) => {
    setDateFilter(value)
    setCurrentPage(1)
    if (query) {
      performSearch(query, searchType, 1)
    }
  }

  const handleLoadMore = () => {
    console.log("load more clicked")
    if (query && hasMoreResults && !loading && isMountedRef.current) {
      performSearch(query, searchType, currentPage + 1)
    }
  }

  const handleImageClick = (image: ImageResult) => {
    console.log(image)
    setSelectedImage(image)
    setIsModalOpen(true)
  }

  const handleCloseModal = () => {
    setIsModalOpen(false)
    setSelectedImage(null)
  }

  const handleVideoClick = (video: VideoResult) => {
    setSelectedVideo(video)
    setIsVideoModalOpen(true)
  }

  const handleCloseVideoModal = () => {
    setIsVideoModalOpen(false)
    setSelectedVideo(null)
  }

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <header className="sticky top-0 z-50 w-full border-b border-border bg-background/95 backdrop-blur-sm">
        <div className="max-w-7xl mx-auto px-4 sm:px-6">

          {/* --- Top Row: Logo & Search --- */}
          <div className="flex flex-col md:flex-row items-center gap-4 lg:gap-4 py-4">
            <div className="flex items-center justify-between w-full md:w-auto">
              {/* <div className="lg:hidden"> */}

              <LogoWithText size="sm" />
              {/* </div> */}
              {/* Mobile-only Settings shortcut */}
              <div className="md:hidden">
                {/* <SettingsModal iconOnly /> */}
                <ActionBar />
              </div>
            </div>

            {/* <div className="relative w-full md:flex-1 md:max-w-2xl group"> */}
            <div className="relative w-full md:flex-1 group ">

              <div className="w-full relative isolate z-10 flex justify-between gap-2">
                <div className="w-full lg:max-w-3xl">
                  <SearchWithAutocomplete
                    value={query}
                    onChange={setQuery}
                    onSubmit={handleSearch}
                    placeholder="Search for anything"
                    size="lg"
                    showButton={true}
                    buttonText="Search"
                    className="placeholder:text-gray-400"
                  />
                </div>
                <div className="hidden md:block">
                  <ActionBar />
                </div>

              </div>

              {/* <div className="flex items-center border border-border rounded-2xl px-4 py-2.5 transition-all focus-within:ring-2 focus-within:ring-ring focus-within:bg-background">
                <Search className="w-4 h-4 text-muted-foreground mr-2" />
                <input
                  value={query}
                  onChange={(e) => setQuery(e.target.value)}
                  placeholder="Search anything..."
                  className="flex-1 bg-transparent border-none outline-none text-sm text-foreground"
                />
                <button
                  type="submit"
                  className="flex rounded-2xl items-center justify-center bg-primary text-primary-foreground hover:opacity-90 px-4 py-1.5 text-xs font-bold transition-all active:scale-95"
                  style={{
                    height: '100%',
                    position: 'absolute',
                    right: 0,
                        borderRadius: '0 12px 12px 0px'
                  }}
                >
                  <span>Search</span>
                </button>

                 <kbd className="hidden sm:inline-flex h-5 select-none items-center gap-1 rounded border border-border bg-muted px-1.5 font-mono text-[10px] font-medium text-muted-foreground">
                  <span className="text-xs">⌘</span>K
              </kbd>
              </div> */}
            </div>
          </div>

          {/* --- Bottom Row: Navigation & Tools --- */}
          <div className="flex items-center justify-between gap-2 pb-4 overflow-x-auto no-scrollbar max-w-7xl mx-auto">

            {/* Main Navigation Tabs */}
            <nav className="flex items-center gap-1 bg-muted/20 p-1 rounded-xl">
              {tab.map((type) => (
                <button
                  key={type.id}
                  onClick={() => handleTypeChange(type.id)}
                  className={`flex items-center gap-2 px-3 sm:px-4 py-1.5 rounded-lg text-xs font-medium transition-all ${searchType === type.id
                    ? "bg-primary text-primary-foreground shadow-lg shadow-primary/20"
                    : "text-muted-foreground hover:bg-muted"
                    }`}
                >
                  <type.icon className="w-3.5 h-3.5" />
                  <span>{type.label}</span>
                </button>
              ))}
            </nav>

            {/* Desktop/Mobile Action Buttons */}
            <div className="flex items-center gap-2 shrink-0">
              <button
                onClick={() => setShowFilters(!showFilters)}
                className={`flex items-center gap-2 px-3 py-1.5 rounded-lg text-xs font-medium border border-border transition-all ${showFilters ? "bg-accent text-accent-foreground ring-2 ring-ring/20" : "bg-card text-muted-foreground hover:bg-muted"
                  }`}
              >
                <Filter className="w-3.5 h-3.5" />
                <span className="hidden sm:inline">Filters</span>
                <ChevronDown className={`w-3 h-3 transition-transform ${showFilters ? "rotate-180" : ""}`} />
              </button>

              {/* <div className="hidden md:block h-8 w-px bg-border mx-1" /> */}

              {/* <div className="hidden md:block">
                <SettingsModal />
              </div> */}
            </div>
          </div>

          {/* --- Filters Drawer: Desktop & Mobile Friendly --- */}
          {showFilters && (
            <div className="animate-in fade-in zoom-in-95 duration-200 border-t border-border py-4 grid grid-cols-1 sm:grid-cols-2 md:grid-cols-4 gap-4">
              <div className="space-y-2">
                <label className="text-[10px] font-bold uppercase text-muted-foreground px-1">Content Safety</label>
                <Select value={safeSearch} onValueChange={handleSafeSearchChange}>
                  <SelectTrigger className="w-full h-9 rounded-xl border-border bg-card">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="on">Safe Search: On</SelectItem>
                    <SelectItem value="off">Safe Search: Off</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-2">
                <label className="text-[10px] font-bold uppercase text-muted-foreground px-1">Time Period</label>
                <Select value={dateFilter} onValueChange={handleDateFilterChange}>
                  <SelectTrigger className="w-full h-9 rounded-xl border-border bg-card">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Anytime</SelectItem>
                    <SelectItem value="24h">Past 24 Hours</SelectItem>
                    <SelectItem value="week">Past Week</SelectItem>
                    <SelectItem value="month">Past Month</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </div>
          )}
        </div>
      </header>





      {/* Results */}
      <main className="max-w-7xl mx-auto px-4 py-6">

        {loading ? (
          <div className="flex items-center justify-center py-12">
            <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-accent"></div>
          </div>
        ) : error ? (
          <div className="flex items-center justify-center py-12">
            <div className="text-center">
              <p className="text-muted-foreground mb-2">Search failed</p>
              <p className="text-sm text-muted-foreground">{error}</p>
              <Button
                variant="outline"
                className="mt-4 bg-transparent"
                onClick={() => query && performSearch(query, searchType)}
              >
                Try Again
              </Button>
            </div>
          </div>
        ) : (
              <>

                {searchType === "web" &&
                  <WebResults searchType={searchType} results={results} query={query} totalResults={totalResults} hasMoreResults={hasMoreResults}
                    handleLoadMore={handleLoadMore}
                    loading={loading}
                    aiSearch={aiSearch}
                    performAiSearch={performAiSearch}
                  />
                }

                {searchType === 'images' &&
                <div>
                  <ImageResults
                    searchType={searchType}
                    imageResults={imageResults}
                    query={query}
                    totalResults={totalResults}
                    handleImageClick={handleImageClick}
                    hasMoreResults={hasMoreResults}
                    loading={loading}
                    handleLoadMore={handleLoadMore}
                    selectedImage={selectedImage}
                    isModalOpen={isModalOpen}
                    handleCloseModal={handleCloseModal}

                  />
                  <ImageModal image={selectedImage} isOpen={isModalOpen} onClose={handleCloseModal} />
                  </div>}


                {/* === Video Results === */}
                {searchType === "videos" && videoResults.length === 0 && query && (
                  <div className="text-center py-12">
                    <p className="text-muted-foreground">No video results found for "{query}"</p>
                    <p className="text-sm text-muted-foreground mt-2">
                      Make sure your YouTube API key is configured in environment variables
                </p>
                  </div>
                )}

                {searchType === "videos" && videoResults.length > 0 && (
                  <>
                    <div className="text-sm text-muted-foreground mb-6">About {totalResults.toLocaleString()} results</div>

                    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
                      {videoResults.map((video) => (
                        <Card
                          key={video.id}
                          className="group overflow-hidden hover:shadow-xl transition-all duration-300 border-0 bg-card/50 backdrop-blur-sm cursor-pointer"
                          onClick={() => handleVideoClick(video)}
                        >
                          <div className="aspect-video relative overflow-hidden">
                            <img
                              src={video.thumbnail || "/placeholder.svg"}
                              alt={video.title}
                              className="w-full h-full object-cover group-hover:scale-105 transition-transform duration-300"
                            />
                            <div className="absolute inset-0 flex items-center justify-center bg-black/40 opacity-0 group-hover:opacity-100 transition-all duration-300">
                              <div className="w-16 h-16 bg-red-600 rounded-full flex items-center justify-center shadow-lg transform scale-75 group-hover:scale-100 transition-transform duration-300">
                                <svg className="w-8 h-8 text-white ml-1" fill="currentColor" viewBox="0 0 24 24">
                                  <path d="M8 5v14l11-7z" />
                                </svg>
                              </div>
                            </div>
                            <div className="absolute bottom-2 right-2 bg-black/80 text-white text-xs px-2 py-1 rounded">
                              Video
                        </div>
                          </div>
                          <CardContent className="p-5">
                            <h3 className="font-semibold mb-3 line-clamp-2 text-primary leading-snug">{video.title}</h3>
                            <div className="space-y-2">
                              {video.channelTitle && (
                                <p className="font-medium text-accent text-sm">{video.channelTitle}</p>
                              )}
                              <div className="flex items-center justify-between">
                                <div className="flex items-center gap-1 text-xs text-muted-foreground">
                                  <Calendar className="w-3 h-3" />
                                  <span>{video.date ? new Date(video.date).toLocaleDateString() : ""}</span>
                                </div>
                                <span className="inline-flex items-center gap-1 text-accent hover:text-accent/80 font-medium text-sm transition-colors group-hover:underline decoration-2 underline-offset-2">
                                  <VideoIcon className="w-3 h-3" />
                              Watch
                            </span>
                              </div>
                            </div>
                            {video.description && (
                              <p className="text-sm text-foreground/70 mt-3 line-clamp-2 leading-relaxed">
                                {video.description}
                              </p>
                            )}
                          </CardContent>
                        </Card>
                      ))}
                    </div>

                    {hasMoreResults && (
                      <div className="flex justify-center mt-8">
                        <Button onClick={handleLoadMore} disabled={loading} variant="outline" size="lg">
                          {loading ? "Loading..." : "Load More Videos"}
                        </Button>
                      </div>
                    )}

                    <VideoModal video={selectedVideo} isOpen={isVideoModalOpen} onClose={handleCloseVideoModal} />
                  </>
                )}
              </>
            )}
      </main>
      <Footer />
    </div>
  )
}

export default function SearchPage() {
  return (
    <Suspense fallback={<div>Loading...</div>}>
      <SearchPageContent />
    </Suspense>
  )
}
